# Copyright 1999-2018 Gentoo Foundation
# Distributed under the terms of the GNU General Public License v2

EAPI=6

inherit autotools flag-o-matic pam qmake-utils readme.gentoo-r1 systemd xdg-utils

DESCRIPTION="A lightweight display manager"
HOMEPAGE="https://www.freedesktop.org/wiki/Software/LightDM"
SRC_URI="https://github.com/CanonicalLtd/lightdm/releases/download/${PV}/${P}.tar.xz
	mirror://gentoo/introspection-20110205.m4.tar.bz2"

LICENSE="GPL-3 LGPL-3"
SLOT="0"
KEYWORDS="~amd64 ~arm ~arm64 ~ppc ~ppc64 ~x86"
IUSE="audit +gtk +introspection qt5 +gnome"

COMMON_DEPEND="
	>=dev-libs/glib-2.44.0:2
	dev-libs/libxml2
	virtual/pam
	x11-libs/libX11
	>=x11-libs/libxklavier-5
	audit? ( sys-process/audit )
	gnome? ( sys-apps/accountsservice )
	introspection? ( >=dev-libs/gobject-introspection-1 )
	qt5? (
		dev-qt/qtcore:5
		dev-qt/qtdbus:5
		dev-qt/qtgui:5
	)"
RDEPEND="${COMMON_DEPEND}
	>=sys-auth/pambase-20101024-r2"
DEPEND="${COMMON_DEPEND}
	dev-util/gtk-doc-am
	dev-util/intltool
	sys-devel/gettext
	virtual/pkgconfig
	gnome? ( gnome-base/gnome-common )"
PDEPEND="gtk? ( x11-misc/lightdm-gtk-greeter )"

DOCS=( NEWS )
RESTRICT="test"

src_prepare() {
	xdg_environment_reset

	sed -i -e 's:getgroups:lightdm_&:' tests/src/libsystem.c || die #412369
	sed -i -e '/minimum-uid/s:500:1000:' data/users.conf || die

	einfo "Fixing the session-wrapper variable in lightdm.conf"
	sed -i -e \
		"/^#session-wrapper/s@^.*@session-wrapper=/etc/${PN}/Xsession@" \
		data/lightdm.conf || die "Failed to fix lightdm.conf"

	# use correct version of qmake. bug #566950
	sed \
		-e "/AC_CHECK_TOOLS(MOC5/a AC_SUBST(MOC5,$(qt5_get_bindir)/moc)" \
		-i configure.ac || die

	default

	# Remove bogus Makefile statement. This needs to go upstream
	sed -i /"@YELP_HELP_RULES@"/d help/Makefile.am || die
	if has_version dev-libs/gobject-introspection; then
		eautoreconf
	else
		AT_M4DIR=${WORKDIR} eautoreconf
	fi
}

src_configure() {
	# Set default values if global vars unset
	local _greeter _session _user
	_greeter=${LIGHTDM_GREETER:=lightdm-gtk-greeter}
	_session=${LIGHTDM_SESSION:=gnome}
	_user=${LIGHTDM_USER:=root}
	# Let user know how lightdm is configured
	einfo "Gentoo configuration"
	einfo "Default greeter: ${_greeter}"
	einfo "Default session: ${_session}"
	einfo "Greeter user: ${_user}"

	use qt5 && append-cxxflags -std=c++11

	# also disable tests because libsystem.c does not build. Tests are
	# restricted so it does not matter anyway.
	local myeconfargs=(
		--localstatedir=/var
		--disable-static
		--disable-tests
		$(use_enable audit libaudit)
		$(use_enable introspection)
		--disable-liblightdm-qt
		$(use_enable qt5 liblightdm-qt5)
		--with-user-session=${_session}
		--with-greeter-session=${_greeter}
		--with-greeter-user=${_user}
	)
	econf "${myeconfargs[@]}"
}

src_install() {
	default

	# Delete apparmor profiles because they only work with Ubuntu's
	# apparmor package. Bug #494426
	if [[ -d ${ED%/}/etc/apparmor.d ]]; then
		rm -r "${ED%/}/etc/apparmor.d" || die \
			"Failed to remove apparmor profiles"
	fi

	insinto /etc/${PN}
	doins data/{${PN},keys}.conf
	doins "${FILESDIR}"/Xsession
	fperms +x /etc/${PN}/Xsession
	# /var/lib/lightdm-data could be useful. Bug #522228
	dodir /var/lib/lightdm-data

	find "${ED}" \( -name '*.a' -o -name "*.la" \) -delete || die
	rm -rf "${ED%/}"/etc/init

	# Remove existing pam file. We will build a new one. Bug #524792
	rm -rf "${ED%/}"/etc/pam.d/${PN}{,-greeter}
	pamd_mimic system-local-login ${PN} auth account password session #372229
	pamd_mimic system-local-login ${PN}-greeter auth account password session #372229
	dopamd "${FILESDIR}"/${PN}-autologin #390863, #423163

	readme.gentoo_create_doc

	systemd_dounit "${FILESDIR}/${PN}.service"
}

pkg_postinst() {
	systemd_reenable "${PN}.service"
}
